<?php
// 🚀 OPTIMIZACIÓN: Output buffering y compresión
ob_start('ob_gzhandler');

// 🔐 PROTECCIÓN CON LOGIN
require_once 'auth.php';
requerirAutenticacion();

$usuario_actual = obtenerUsuarioActual();

// 🚀 Headers de caché del navegador (5 min)
header('Cache-Control: private, max-age=300');
header('Vary: Accept-Encoding');
?>
<!DOCTYPE html>
<html lang="es">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sistema Organizado de Tarjetas</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 15px;
            line-height: 1.6;
        }

        .container {
            max-width: 1600px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            box-shadow: 0 25px 50px rgba(0, 0, 0, 0.15);
            overflow: hidden;
        }

        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px 30px;
            text-align: center;
            position: relative;
        }

        .header h1 {
            font-size: clamp(2rem, 5vw, 3rem);
            margin-bottom: 15px;
            font-weight: 700;
        }

        .header p {
            opacity: 0.95;
            font-size: clamp(1rem, 2.5vw, 1.2rem);
        }

        .user-info {
            position: absolute;
            top: 20px;
            right: 20px;
            display: flex;
            align-items: center;
            gap: 15px;
            background: rgba(255, 255, 255, 0.2);
            padding: 10px 20px;
            border-radius: 25px;
            backdrop-filter: blur(10px);
        }

        .user-name {
            font-weight: 600;
            font-size: 0.95em;
        }

        .btn-logout {
            background: rgba(255, 255, 255, 0.3);
            color: white;
            border: 2px solid rgba(255, 255, 255, 0.5);
            padding: 8px 18px;
            border-radius: 20px;
            cursor: pointer;
            font-weight: 600;
            font-size: 0.9em;
            transition: all 0.3s ease;
            text-decoration: none;
        }

        .btn-logout:hover {
            background: rgba(255, 255, 255, 0.4);
            border-color: white;
            transform: translateY(-2px);
        }

        @media (max-width: 768px) {
            .user-info {
                position: static;
                margin: 20px auto 0;
                width: fit-content;
            }
        }

        .content {
            padding: 40px 30px;
        }

        .stats-bar {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 40px;
        }

        .stat-box {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding: 25px;
            border-radius: 15px;
            text-align: center;
            border-left: 5px solid #667eea;
            transition: transform 0.3s ease;
        }

        .stat-box:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
        }

        .stat-number {
            font-size: 2.5em;
            font-weight: 800;
            color: #667eea;
            margin-bottom: 5px;
        }

        .stat-label {
            color: #495057;
            font-weight: 600;
            font-size: 1em;
        }

        .search-section {
            margin-bottom: 30px;
        }

        .search-box {
            width: 100%;
            padding: 18px 25px;
            border: 3px solid #e9ecef;
            border-radius: 50px;
            font-size: 1.1em;
            transition: all 0.3s ease;
            background: white;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
        }

        .search-box:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 5px 20px rgba(102, 126, 234, 0.2);
            transform: scale(1.02);
        }

        .filters-bar {
            display: flex;
            gap: 15px;
            margin-bottom: 30px;
            flex-wrap: wrap;
            justify-content: center;
        }

        .filter-btn {
            background: white;
            border: 2px solid #e9ecef;
            padding: 12px 25px;
            border-radius: 25px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s ease;
            color: #495057;
        }

        .filter-btn:hover {
            border-color: #667eea;
            color: #667eea;
            transform: translateY(-2px);
        }

        .filter-btn.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-color: transparent;
        }

        /* Acordeón de Bancos */
        .bank-accordion {
            margin-bottom: 20px;
        }

        .bank-header {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding: 20px 25px;
            border-radius: 15px;
            cursor: pointer;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: all 0.3s ease;
            border: 2px solid transparent;
            margin-bottom: 5px;
        }

        .bank-header:hover {
            background: linear-gradient(135deg, #e9ecef 0%, #dee2e6 100%);
            border-color: #667eea;
            transform: translateX(5px);
        }

        .bank-header.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 15px 15px 0 0;
        }

        .bank-name {
            font-size: 1.3em;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .bank-count {
            background: rgba(255, 255, 255, 0.9);
            color: #667eea;
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 0.9em;
            font-weight: 700;
        }

        .bank-header.active .bank-count {
            background: rgba(255, 255, 255, 0.2);
            color: white;
        }

        .bank-arrow {
            font-size: 1.5em;
            transition: transform 0.3s ease;
        }

        .bank-header.active .bank-arrow {
            transform: rotate(180deg);
        }

        .bank-content {
            display: none;
            background: white;
            border: 2px solid #e9ecef;
            border-top: none;
            border-radius: 0 0 15px 15px;
        }

        .bank-content.active {
            display: block;
            padding: 25px;
        }

        /* Tabs de Tipo de Tarjeta */
        .type-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 25px;
            border-bottom: 3px solid #e9ecef;
            padding-bottom: 5px;
        }

        .type-tab {
            padding: 12px 30px;
            border-radius: 10px 10px 0 0;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s ease;
            color: #6c757d;
            background: #f8f9fa;
            border: 2px solid #e9ecef;
            border-bottom: none;
        }

        .type-tab:hover {
            background: #e9ecef;
            color: #495057;
        }

        .type-tab.active {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            border-color: transparent;
            transform: translateY(-2px);
        }

        .type-content {
            display: none;
        }

        .type-content.active {
            display: block;
        }

        /* Secciones de Marca */
        .brand-section {
            margin-bottom: 30px;
        }

        .brand-header {
            background: linear-gradient(135deg, #ffc107 0%, #ff9800 100%);
            color: white;
            padding: 15px 20px;
            border-radius: 10px;
            font-weight: 700;
            font-size: 1.2em;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .brand-header.visa {
            background: linear-gradient(135deg, #1a1f71 0%, #0d47a1 100%);
        }

        .brand-header.mastercard {
            background: linear-gradient(135deg, #eb001b 0%, #ff5f00 100%);
        }

        .brand-header.amex {
            background: linear-gradient(135deg, #006fcf 0%, #00c2ff 100%);
        }

        .brand-header.discover {
            background: linear-gradient(135deg, #ff6000 0%, #ff9000 100%);
        }

        /* Tarjetas */
        .cards-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }

        .card-item {
            transition: all 0.3s ease;
        }

        @media (min-width: 1400px) {
            .cards-grid {
                grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
            }
        }

        @media (min-width: 1800px) {
            .cards-grid {
                grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            }
        }

        .card-item {
            background: white;
            border: 2px solid #f1f3f4;
            border-radius: 15px;
            padding: 20px;
            transition: transform 0.15s ease, box-shadow 0.15s ease, border-color 0.15s ease;
            position: relative;
            cursor: pointer;
            contain: layout style;
        }

        .card-item:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.08);
            border-color: #667eea;
        }

        .card-item.used {
            border-color: #dc3545;
            background: linear-gradient(135deg, #ffe6e6 0%, #fff0f0 100%);
            opacity: 0.7;
        }

        .card-item.used .card-number {
            background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
            color: #721c24;
        }

        .used-badge {
            position: absolute;
            top: 10px;
            left: 10px;
            background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
            color: white;
            padding: 5px 12px;
            border-radius: 15px;
            font-size: 0.8em;
            font-weight: 600;
            z-index: 10;
            display: none;
        }

        .card-item.used .used-badge {
            display: block;
        }


        .card-item.selecting {
            background: linear-gradient(135deg, #fff3cd 0%, #ffe69c 100%);
            border-color: #ffc107;
            transform: scale(0.98);
            transition: all 0.15s ease;
        }

        .card-item:hover:not(.selected):not(.used) {
            cursor: pointer;
        }

        .card-item.selected {
            background: linear-gradient(135deg, #cfe2ff 0%, #9ec5fe 100%);
            border-color: #0d6efd;
            border-width: 3px;
            transform: scale(1.02);
            box-shadow: 0 8px 25px rgba(13, 110, 253, 0.4);
        }

        .card-item.selected::after {
            content: '✓';
            position: absolute;
            top: 5px;
            left: 5px;
            background: #0d6efd;
            color: white;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            font-size: 1.2em;
            z-index: 100;
            box-shadow: 0 2px 8px rgba(0,0,0,0.3);
        }

        .selection-counter {
            position: fixed;
            bottom: 80px;
            right: 20px;
            background: linear-gradient(135deg, #0d6efd 0%, #0a58ca 100%);
            color: white;
            padding: 15px 25px;
            border-radius: 25px;
            font-weight: 700;
            font-size: 1.1em;
            z-index: 1001;
            box-shadow: 0 8px 25px rgba(13, 110, 253, 0.5);
            display: none;
            animation: bounceIn 0.3s ease;
        }

        @keyframes bounceIn {
            0% { transform: scale(0.3); opacity: 0; }
            50% { transform: scale(1.1); }
            100% { transform: scale(1); opacity: 1; }
        }

        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.05); }
            100% { transform: scale(1); }
        }

        .selection-counter.show {
            display: block;
        }

        .selection-toolbar {
            position: fixed;
            bottom: 20px;
            left: 50%;
            transform: translateX(-50%);
            background: white;
            padding: 20px 30px;
            border-radius: 50px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
            z-index: 1001;
            display: none;
            gap: 15px;
            align-items: center;
        }

        .selection-toolbar.show {
            display: flex;
            animation: slideUp 0.3s ease;
        }

        @keyframes slideUp {
            from { transform: translateX(-50%) translateY(100px); opacity: 0; }
            to { transform: translateX(-50%) translateY(0); opacity: 1; }
        }

        .selection-toolbar button {
            background: linear-gradient(135deg, #0d6efd 0%, #0a58ca 100%);
            color: white;
            border: none;
            padding: 12px 25px;
            border-radius: 25px;
            cursor: pointer;
            font-weight: 700;
            font-size: 1em;
            transition: all 0.3s ease;
        }

        .selection-toolbar button:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(13, 110, 253, 0.4);
        }

        .selection-toolbar button.cancel {
            background: linear-gradient(135deg, #6c757d 0%, #495057 100%);
        }

        .selection-toolbar .count-badge {
            background: linear-gradient(135deg, #0d6efd 0%, #0a58ca 100%);
            color: white;
            padding: 8px 20px;
            border-radius: 20px;
            font-weight: 700;
            font-size: 1.1em;
        }

        /* Deshabilitar selección de texto durante arrastre */
        body.selecting {
            user-select: none;
            -webkit-user-select: none;
            -moz-user-select: none;
            -ms-user-select: none;
        }

        .card-bin {
            position: absolute;
            top: 10px;
            right: 10px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 0.85em;
            font-weight: 700;
            font-family: 'Courier New', monospace;
        }

        .card-number {
            font-family: 'Fira Code', 'Courier New', monospace;
            font-size: 1.4em;
            font-weight: 700;
            color: #2c3e50;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 15px;
            word-break: break-all;
        }

        .card-details {
            display: flex;
            flex-direction: column;
            gap: 10px;
            margin-top: 10px;
        }

        @media (min-width: 600px) {
            .card-details {
                display: grid;
                grid-template-columns: repeat(2, 1fr);
            }
        }

        .card-detail {
            display: flex;
            flex-direction: column;
            gap: 3px;
        }

        .card-detail-label {
            font-size: 0.85em;
            color: #6c757d;
            font-weight: 600;
        }

        .card-detail-value {
            font-size: 1em;
            color: #2c3e50;
            font-weight: 500;
        }

        .copy-btn {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            border: none;
            border-radius: 10px;
            padding: 12px 20px;
            cursor: pointer;
            font-size: 0.95em;
            font-weight: 600;
            transition: all 0.3s ease;
            width: 100%;
            margin-top: 15px;
        }

        .copy-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 20px rgba(40, 167, 69, 0.3);
        }

        .copy-btn.copied {
            background: linear-gradient(135deg, #007bff 0%, #6610f2 100%);
        }

        .actions-bar {
            background: #f8f9fa;
            padding: 25px;
            border-radius: 15px;
            text-align: center;
            margin-bottom: 30px;
        }

        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px 30px;
            border: none;
            border-radius: 30px;
            cursor: pointer;
            font-size: 1em;
            font-weight: 600;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-block;
            margin: 8px 10px;
            min-width: 140px;
        }

        .btn:hover {
            transform: translateY(-3px) scale(1.05);
            box-shadow: 0 10px 25px rgba(102, 126, 234, 0.4);
        }

        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        /* Animación deshabilitada para mejor rendimiento con muchas tarjetas */

        /* Responsive */
        @media (max-width: 768px) {
            .cards-grid {
                grid-template-columns: 1fr;
            }

            .bank-name {
                font-size: 1.1em;
            }

            .type-tabs {
                flex-wrap: wrap;
            }

            .filters-bar {
                flex-direction: column;
            }

            .filter-btn {
                width: 100%;
            }
        }

        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #6c757d;
        }

        .empty-state-icon {
            font-size: 4em;
            margin-bottom: 20px;
        }

        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(102, 126, 234, 0.95);
            z-index: 9999;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            color: white;
        }

        .loading-spinner {
            width: 60px;
            height: 60px;
            border: 6px solid rgba(255, 255, 255, 0.3);
            border-top: 6px solid white;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-bottom: 30px;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .loading-text {
            font-size: 1.5em;
            font-weight: 700;
            margin-bottom: 10px;
        }

        .loading-subtext {
            font-size: 1em;
            opacity: 0.9;
        }

        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
            background: #28a745;
            color: white;
            padding: 15px 25px;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
            z-index: 9999;
            animation: slideIn 0.3s ease-out;
        }

        @keyframes slideIn {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
    </style>
</head>

<body>
    <!-- Loading overlay -->
    <div class="loading-overlay" id="loadingOverlay">
        <div class="loading-spinner"></div>
        <div class="loading-text">Cargando tarjetas...</div>
        <div class="loading-subtext">Organizando por banco, tipo y marca</div>
    </div>

    <!-- Selection Counter -->
    <div class="selection-counter" id="selectionCounter">
        📋 0 seleccionadas
    </div>

    <!-- Selection Toolbar -->
    <div class="selection-toolbar" id="selectionToolbar">
        <div class="count-badge">
            <span id="toolbarCount">0</span> tarjetas
        </div>
        <button onclick="copySelectedCards()">
            📋 Copiar Todas
        </button>
        <button onclick="clearSelection()" class="cancel">
            ✖️ Cancelar
        </button>
    </div>

    <div class="container">
        <div class="header">
            <div class="user-info">
                <span class="user-name">👤 <?php echo htmlspecialchars($usuario_actual['nombre']); ?></span>
                <form method="POST" action="auth.php" style="margin: 0;">
                    <input type="hidden" name="action" value="logout">
                    <button type="submit" class="btn-logout">🚪 Salir</button>
                </form>
            </div>
            <h1>🏦 Sistema Organizado de Tarjetas</h1>
            <p>Navegación intuitiva por banco, tipo y marca</p>
        </div>

        <div class="content">
            <?php
            require_once 'CreditCardExtractor.php';

            try {
                $extractor = new CreditCardExtractor();
                
                // 🚀 Si se pide refresh, limpiar caché
                if (isset($_GET['refresh'])) {
                    $extractor->limpiarCache();
                }
                
                $datos_extraidos      = $extractor->extraerTodasLasTablas();
                $tarjetas_formateadas = $extractor->formatearTarjetas($datos_extraidos);

                // Función para detectar la marca de la tarjeta por el primer dígito
                function detectarMarca($numero)
                {
                    $primer_digito = substr($numero, 0, 1);
                    $primeros_dos  = substr($numero, 0, 2);

                    if ($primer_digito === '4') return 'Visa';
                    if ($primeros_dos >= '51' && $primeros_dos <= '55') return 'Mastercard';
                    if ($primeros_dos === '34' || $primeros_dos === '37') return 'Amex';
                    if ($primeros_dos === '60' || $primeros_dos === '65') return 'Discover';
                    return 'Otra';
                }

                // Función para detectar tipo de tarjeta
                function detectarTipo($numero, $bankname = '')
                {
                    $primeros_seis = substr($numero, 0, 6);
                    $primeros_cuatro = substr($numero, 0, 4);
                    
                    // Rangos específicos de débito conocidos
                    $rangos_debito = [
                        // Visa Débito
                        '402360', '402361', '403626', '405919', '407197', '407278', '408849',
                        '413449', '414720', '417633', '419740', '423460', '424631', '424680',
                        '428671', '430108', '431380', '434107', '434645', '446286', '447031',
                        '451416', '455187', '456619', '465375', '474491', '476142', '476143',
                        '482387', '483010', '484406', '491032',
                        
                        // Mastercard Débito
                        '516400', '516500', '516600', '516700', '516800', '516900',
                        '517000', '517100', '517200', '517300', '517400', '517500',
                        '517600', '517700', '517800', '517900',
                    ];
                    
                    // Verificar si el BIN está en rangos de débito
                    if (in_array($primeros_seis, $rangos_debito)) {
                        return 'Débito';
                    }
                    
                    // Buscar palabras clave en el nombre del banco
                    $bankname_lower = strtolower($bankname);
                    if (strpos($bankname_lower, 'debit') !== false || 
                        strpos($bankname_lower, 'débito') !== false ||
                        strpos($bankname_lower, 'debito') !== false) {
                        return 'Débito';
                    }
                    
                    // Por defecto, asumir crédito
                    return 'Crédito';
                }

                // Organizar tarjetas por banco, tipo y marca
                $tarjetas_organizadas = [];
                foreach ($tarjetas_formateadas as $tarjeta) {
                    $banco  = $tarjeta['datos_raw']['bankname'] ?? 'Sin Especificar';
                    $numero = explode('|', $tarjeta['formato'])[0];
                    $marca  = detectarMarca($numero);
                    $tipo   = detectarTipo($numero, $banco);

                    if (!isset($tarjetas_organizadas[$banco])) {
                        $tarjetas_organizadas[$banco] = [
                            'Crédito' => [],
                            'Débito'  => []
                        ];
                    }

                    if (!isset($tarjetas_organizadas[$banco][$tipo][$marca])) {
                        $tarjetas_organizadas[$banco][$tipo][$marca] = [];
                    }

                    $tarjetas_organizadas[$banco][$tipo][$marca][] = $tarjeta;
                }

                // Ordenar bancos alfabéticamente
                ksort($tarjetas_organizadas);

                // Calcular estadísticas
                $total_tarjetas = count($tarjetas_formateadas);
                $total_bancos   = count($tarjetas_organizadas);
                
                $marcas_count = [];
                foreach ($tarjetas_formateadas as $t) {
                    $num = explode('|', $t['formato'])[0];
                    $marca = detectarMarca($num);
                    $marcas_count[$marca] = ($marcas_count[$marca] ?? 0) + 1;
                }

                // Obtener nombre de la DB
                $db_name = $extractor->obtenerDatabaseName();
                
                // Mostrar estadísticas
                echo '<div class="stats-bar">';
                echo '<div class="stat-box">';
                echo '<div class="stat-number">' . $total_tarjetas . '</div>';
                echo '<div class="stat-label">💳 Total Tarjetas</div>';
                echo '</div>';

                echo '<div class="stat-box">';
                echo '<div class="stat-number">' . $total_bancos . '</div>';
                echo '<div class="stat-label">🏦 Bancos</div>';
                echo '</div>';

                echo '<div class="stat-box">';
                echo '<div class="stat-number">' . ($marcas_count['Visa'] ?? 0) . '</div>';
                echo '<div class="stat-label">💙 Visa</div>';
                echo '</div>';

                echo '<div class="stat-box">';
                echo '<div class="stat-number">' . ($marcas_count['Mastercard'] ?? 0) . '</div>';
                echo '<div class="stat-label">🔴 Mastercard</div>';
                echo '</div>';

                echo '<div class="stat-box" style="border-left-color: #28a745;">';
                echo '<div class="stat-number" style="color: #28a745; font-size: 1.5em;">💾</div>';
                echo '<div class="stat-label">' . htmlspecialchars($db_name) . '</div>';
                echo '</div>';

                echo '<div class="stat-box">';
                echo '<div class="stat-number">' . date('H:i') . '</div>';
                echo '<div class="stat-label">🕒 Actualización</div>';
                echo '</div>';
                echo '</div>';

                // Contar tarjetas con dirección (considerar múltiples campos)
                $con_direccion = 0;
                foreach ($tarjetas_formateadas as $tarjeta) {
                    $tiene_dir = !empty($tarjeta['datos_raw']['dir'] ?? '');
                    $tiene_ciudad = !empty($tarjeta['datos_raw']['ciudad'] ?? '');
                    $tiene_pais = !empty($tarjeta['datos_raw']['pais'] ?? '');
                    if ($tiene_dir || $tiene_ciudad || $tiene_pais) {
                        $con_direccion++;
                    }
                }
                
                // Mensaje informativo
                echo '<div style="background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%); border: 2px solid #b8dabd; color: #155724; padding: 20px 25px; border-radius: 15px; margin-bottom: 30px; font-weight: 500;">';
                echo '✅ <strong>Sistema Actualizado:</strong> Se cargaron <strong>' . number_format($total_tarjetas) . ' tarjetas válidas</strong> organizadas por banco, tipo y marca. ';
                echo '💾 <strong>Base de Datos:</strong> ' . htmlspecialchars($extractor->obtenerDatabaseName()) . '. ';
                if ($con_direccion > 0) {
                    echo '📍 <strong>' . $con_direccion . ' tarjetas</strong> con dirección. ';
                }
                echo '<br>✨ <strong>NUEVO:</strong> Las tarjetas NO USADAS aparecen primero, las USADAS se mueven al final automáticamente.';
                echo '</div>';

                // Tips de uso
                echo '<div style="background: linear-gradient(135deg, #cfe2ff 0%, #9ec5fe 100%); border: 2px solid #6ea8fe; color: #084298; padding: 20px 25px; border-radius: 15px; margin-bottom: 30px;">';
                echo '<div style="font-weight: 700; font-size: 1.1em; margin-bottom: 10px;">💡 Tips de Uso Rápido:</div>';
                echo '<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap: 10px;">';
                echo '<div>🖱️ <strong>Arrastrar:</strong> Mantén click y arrastra sobre las tarjetas para seleccionar múltiples</div>';
                echo '<div>✨ <strong>Orden:</strong> Las NO USADAS aparecen primero, las USADAS van al final</div>';
                echo '<div>📋 <strong>Copiar:</strong> Al soltar el mouse, se copian automáticamente en formato cc|mm/yy|cvv</div>';
                echo '<div>⌨️ <strong>Ctrl+A:</strong> Seleccionar todas las visibles (solo no usadas si filtras)</div>';
                echo '<div>👁️ <strong>Filtro:</strong> Usa "Mostrar Solo Nuevas" para ocultar completamente las usadas</div>';
                echo '<div>🔄 <strong>Resetear:</strong> Limpia todas las marcas de usadas cuando necesites</div>';
                echo '</div>';
                echo '</div>';

                // Barra de acciones
                echo '<div class="actions-bar">';
                echo '<h3 style="margin-bottom: 20px; color: #495057;">🚀 Acciones Rápidas</h3>';
                echo '<button onclick="expandAll()" class="btn">📂 Expandir Todo</button>';
                echo '<button onclick="collapseAll()" class="btn">📁 Colapsar Todo</button>';
                echo '<button onclick="selectAllVisible()" class="btn" style="background: linear-gradient(135deg, #0d6efd 0%, #0a58ca 100%);">✅ Seleccionar Todas</button>';
                echo '<button onclick="toggleUsedCards()" class="btn" id="toggleUsedBtn">👁️ Mostrar Solo Nuevas</button>';
                echo '<button onclick="resetUsedCards()" class="btn">🔄 Resetear Usadas</button>';
                echo '<button onclick="showUsedCardsStats()" class="btn">📊 Estadísticas de Uso</button>';
                echo '<a href="?download=json" class="btn">📥 Descargar JSON</a>';
                echo '<a href="?refresh=1" class="btn" style="background: linear-gradient(135deg, #dc3545 0%, #b02a37 100%);">🔄 Refrescar (Limpiar Caché)</a>';
                echo '</div>';
                
                // Mostrar si se usa caché
                $cache_file = __DIR__ . '/cache_tarjetas.json';
                if (file_exists($cache_file) && !isset($_GET['refresh'])) {
                    $cache_time = filemtime($cache_file);
                    $cache_age = time() - $cache_time;
                    echo '<div style="background: linear-gradient(135deg, #ffc107 0%, #ffca2c 100%); color: #212529; padding: 10px 20px; border-radius: 10px; margin-bottom: 20px; font-size: 0.9em; text-align: center;">';
                    echo '⚡ <strong>Carga rápida desde caché</strong> (guardado hace ' . $cache_age . ' segundos) - ';
                    echo '<a href="?refresh=1" style="color: #0d6efd; text-decoration: underline;">Click aquí para refrescar datos</a>';
                    echo '</div>';
                }

                // Búsqueda
                echo '<div class="search-section">';
                echo '<input type="text" class="search-box" id="searchBox" placeholder="🔍 Buscar por banco, número, titular..." onkeyup="filtrarTarjetas()">';
                echo '</div>';

                // Filtros
                echo '<div class="filters-bar">';
                echo '<button class="filter-btn active" onclick="filtrarPorMarca(\'all\', this)">Todas</button>';
                echo '<button class="filter-btn" onclick="filtrarPorMarca(\'Visa\', this)">💙 Visa</button>';
                echo '<button class="filter-btn" onclick="filtrarPorMarca(\'Mastercard\', this)">🔴 Mastercard</button>';
                echo '<button class="filter-btn" onclick="filtrarPorMarca(\'Amex\', this)">💚 Amex</button>';
                echo '<button class="filter-btn" onclick="filtrarPorMarca(\'Discover\', this)">🧡 Discover</button>';
                echo '</div>';

                // Mostrar tarjetas organizadas por banco
                echo '<div id="bancos-container">';
                foreach ($tarjetas_organizadas as $nombre_banco => $tipos) {
                    $total_banco = 0;
                    foreach ($tipos as $tipo => $marcas) {
                        foreach ($marcas as $marca => $tarjetas) {
                            $total_banco += count($tarjetas);
                        }
                    }

                    echo '<div class="bank-accordion" data-banco="' . htmlspecialchars($nombre_banco) . '">';
                    echo '<div class="bank-header" onclick="toggleBank(this)">';
                    echo '<div class="bank-name">';
                    echo '<span>🏦</span>';
                    echo '<span>' . htmlspecialchars($nombre_banco) . '</span>';
                    echo '<span class="bank-count">' . $total_banco . ' tarjetas</span>';
                    echo '</div>';
                    echo '<div class="bank-arrow">▼</div>';
                    echo '</div>';

                    echo '<div class="bank-content">';
                    
                    // Tabs de tipo (Crédito/Débito)
                    echo '<div class="type-tabs">';
                    $tipo_index = 0;
                    foreach ($tipos as $tipo => $marcas) {
                        if (!empty($marcas)) {
                            $active_class = $tipo_index === 0 ? 'active' : '';
                            $tipo_count = 0;
                            foreach ($marcas as $marca => $tarjetas) {
                                $tipo_count += count($tarjetas);
                            }
                            echo '<div class="type-tab ' . $active_class . '" onclick="toggleType(this)" data-tipo="' . $tipo . '">';
                            echo $tipo . ' (' . $tipo_count . ')';
                            echo '</div>';
                            $tipo_index++;
                        }
                    }
                    echo '</div>';

                    // Contenido de cada tipo
                    $tipo_index = 0;
                    foreach ($tipos as $tipo => $marcas) {
                        if (!empty($marcas)) {
                            $active_class = $tipo_index === 0 ? 'active' : '';
                            echo '<div class="type-content ' . $active_class . '" data-tipo="' . $tipo . '">';

                            // Mostrar por marca
                            foreach ($marcas as $marca => $tarjetas) {
                                echo '<div class="brand-section" data-marca="' . $marca . '">';
                                $marca_class = strtolower($marca);
                                $marca_icons = [
                                    'Visa' => '💙',
                                    'Mastercard' => '🔴',
                                    'Amex' => '💚',
                                    'Discover' => '🧡',
                                    'Otra' => '💳'
                                ];
                                echo '<div class="brand-header ' . $marca_class . '">';
                                echo ($marca_icons[$marca] ?? '💳') . ' ' . $marca . ' - ' . count($tarjetas) . ' tarjetas';
                                echo '</div>';

                                echo '<div class="cards-grid">';
                                foreach ($tarjetas as $tarjeta) {
                                    $formato_completo = $tarjeta['formato'];
                                    $formato_basico   = explode(' - ', $formato_completo)[0];
                                    $partes           = explode('|', $formato_basico);
                                    $numero           = $partes[0];
                                    $fecha            = $partes[1];
                                    $cvv              = $partes[2];
                                    $bin              = substr($numero, 0, 6);
                                    $titular          = htmlspecialchars($tarjeta['titular']);
                                    $card_id          = 'card_' . md5($formato_completo);

                                    echo '<div class="card-item" id="' . $card_id . '" data-search="' . strtolower($titular . ' ' . $numero . ' ' . $nombre_banco) . '" data-marca="' . $marca . '">';
                                    echo '<div class="used-badge">❌ USADA</div>';
                                    echo '<div class="card-bin">BIN: ' . $bin . '</div>';
                                    echo '<div class="card-number">' . $numero . '</div>';
                                    echo '<div class="card-details">';
                                    
                                    echo '<div class="card-detail">';
                                    echo '<span class="card-detail-label">📅 Vencimiento</span>';
                                    echo '<span class="card-detail-value">' . htmlspecialchars($fecha) . '</span>';
                                    echo '</div>';

                                    echo '<div class="card-detail">';
                                    echo '<span class="card-detail-label">🔒 CVV</span>';
                                    echo '<span class="card-detail-value">' . htmlspecialchars($cvv) . '</span>';
                                    echo '</div>';

                                    echo '<div class="card-detail">';
                                    echo '<span class="card-detail-label">🏦 Banco</span>';
                                    echo '<span class="card-detail-value">' . htmlspecialchars($tarjeta['datos_raw']['bankname'] ?? 'N/A') . '</span>';
                                    echo '</div>';

                                    // Mostrar dirección si está disponible (buscar en múltiples campos)
                                    $direccion = '';
                                    $ciudad = htmlspecialchars($tarjeta['datos_raw']['ciudad'] ?? '');
                                    $pais = htmlspecialchars($tarjeta['datos_raw']['pais'] ?? '');
                                    $dir_base = htmlspecialchars($tarjeta['datos_raw']['dir'] ?? '');
                                    
                                    // Construir dirección completa
                                    $partes_direccion = array_filter([$dir_base, $ciudad, $pais]);
                                    $direccion = implode(', ', $partes_direccion);
                                    
                                    if ($direccion) {
                                        echo '<div class="card-detail">';
                                        echo '<span class="card-detail-label">📍 Dirección</span>';
                                        echo '<span class="card-detail-value">' . $direccion . '</span>';
                                        echo '</div>';
                                    }

                                    // Mostrar base de datos de origen
                                    $db_origen = $extractor->obtenerDatabaseName();
                                    echo '<div class="card-detail">';
                                    echo '<span class="card-detail-label">💾 Base de Datos</span>';
                                    echo '<span class="card-detail-value">' . htmlspecialchars($db_origen) . '</span>';
                                    echo '</div>';

                                    // Mostrar tabla de origen
                                    $tabla_origen = htmlspecialchars($tarjeta['tabla_origen']);
                                    echo '<div class="card-detail">';
                                    echo '<span class="card-detail-label">📊 Tabla Origen</span>';
                                    echo '<span class="card-detail-value"><code style="background: #667eea; color: white; padding: 3px 8px; border-radius: 4px; font-size: 0.9em;">' . $tabla_origen . '</code></span>';
                                    echo '</div>';

                                    echo '</div>';
                                    echo '<button class="copy-btn" onclick="copiarTarjeta(\'' . htmlspecialchars($formato_basico) . '\', this, \'' . $card_id . '\')">📋 Copiar</button>';
                                    echo '</div>';
                                }
                                echo '</div>';
                                echo '</div>';
                            }

                            echo '</div>';
                            $tipo_index++;
                        }
                    }

                    echo '</div>';
                    echo '</div>';
                }
                echo '</div>';

                // Descargar JSON
                if (isset($_GET['download']) && $_GET['download'] === 'json') {
                    $json_data = [
                        'fecha_extraccion' => date('Y-m-d H:i:s'),
                        'total_tarjetas'   => count($tarjetas_formateadas),
                        'tarjetas'         => $tarjetas_formateadas
                    ];

                    header('Content-Type: application/json');
                    header('Content-Disposition: attachment; filename="tarjetas_' . date('Y-m-d_H-i-s') . '.json"');
                    echo json_encode($json_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                    exit;
                }

                $extractor->cerrarConexion();

            } catch (Exception $e) {
                echo '<div class="empty-state">';
                echo '<div class="empty-state-icon">❌</div>';
                echo '<h3>Error al cargar los datos</h3>';
                echo '<p>' . htmlspecialchars($e->getMessage()) . '</p>';
                echo '</div>';
            }
            ?>
        </div>
    </div>

    <script>
        // 🚀 OPTIMIZACIÓN: Debounce para evitar ejecutar funciones muy seguido
        function debounce(func, wait) {
            let timeout;
            return function(...args) {
                clearTimeout(timeout);
                timeout = setTimeout(() => func.apply(this, args), wait);
            };
        }

        // Toggle acordeón de banco
        function toggleBank(element) {
            const content = element.nextElementSibling;
            element.classList.toggle('active');
            content.classList.toggle('active');
        }

        // Toggle tabs de tipo
        function toggleType(element) {
            const parent = element.parentElement.parentElement;
            const tipo = element.dataset.tipo;
            
            parent.querySelectorAll('.type-tab').forEach(tab => tab.classList.remove('active'));
            parent.querySelectorAll('.type-content').forEach(content => content.classList.remove('active'));
            
            element.classList.add('active');
            parent.querySelector(`.type-content[data-tipo="${tipo}"]`).classList.add('active');
        }

        // Expandir todos los bancos
        function expandAll() {
            document.querySelectorAll('.bank-header').forEach(header => {
                header.classList.add('active');
                header.nextElementSibling.classList.add('active');
            });
            mostrarNotificacion('✅ Todos los bancos expandidos');
        }

        // Colapsar todos los bancos
        function collapseAll() {
            document.querySelectorAll('.bank-header').forEach(header => {
                header.classList.remove('active');
                header.nextElementSibling.classList.remove('active');
            });
            mostrarNotificacion('✅ Todos los bancos colapsados');
        }

        // 🚀 OPTIMIZACIÓN: Filtrar con debounce y requestAnimationFrame
        let filtroTimeout;
        function filtrarTarjetas() {
            clearTimeout(filtroTimeout);
            filtroTimeout = setTimeout(() => {
                requestAnimationFrame(() => {
                    const filtro = document.getElementById('searchBox').value.toLowerCase();
                    const bancos = document.querySelectorAll('.bank-accordion');

                    bancos.forEach(banco => {
                        let tieneResultados = false;
                        const tarjetas = banco.querySelectorAll('.card-item');

                        tarjetas.forEach(tarjeta => {
                            const texto = tarjeta.dataset.search;
                            const visible = texto.includes(filtro);
                            tarjeta.style.display = visible ? '' : 'none';
                            if (visible) tieneResultados = true;
                        });

                        if (filtro !== '') {
                            banco.style.display = tieneResultados ? '' : 'none';
                            if (tieneResultados) {
                                banco.querySelector('.bank-header').classList.add('active');
                                banco.querySelector('.bank-content').classList.add('active');
                            }
                        } else {
                            banco.style.display = '';
                        }
                    });
                });
            }, 150); // 150ms de debounce
        }

        // Filtrar por marca
        let marcaActual = 'all';
        function filtrarPorMarca(marca, button) {
            marcaActual = marca;
            
            // Actualizar botones activos
            document.querySelectorAll('.filter-btn').forEach(btn => btn.classList.remove('active'));
            button.classList.add('active');

            // Filtrar tarjetas
            const tarjetas = document.querySelectorAll('.card-item');
            tarjetas.forEach(tarjeta => {
                const marcaTarjeta = tarjeta.getAttribute('data-marca');
                if (marca === 'all' || marcaTarjeta === marca) {
                    tarjeta.style.display = 'block';
                } else {
                    tarjeta.style.display = 'none';
                }
            });

            // Ocultar secciones de marca vacías
            document.querySelectorAll('.brand-section').forEach(section => {
                const marcaSection = section.getAttribute('data-marca');
                const cards = section.querySelectorAll('.card-item[style*="display: block"], .card-item:not([style*="display: none"])');
                
                if (marca === 'all' || marcaSection === marca) {
                    section.style.display = 'block';
                } else {
                    section.style.display = 'none';
                }
            });

            mostrarNotificacion(`🔍 Filtrando: ${marca === 'all' ? 'Todas las marcas' : marca}`);
        }

        // Copiar tarjeta y marcar como usada
        function copiarTarjeta(formato, button, cardId) {
            navigator.clipboard.writeText(formato).then(() => {
                const originalText = button.textContent;
                button.textContent = '✅ ¡Copiado!';
                button.classList.add('copied');
                
                // Marcar la tarjeta como usada
                markCardAsUsed(cardId);
                
                mostrarNotificacion('✅ Tarjeta copiada y marcada como USADA: ' + formato);
                
                setTimeout(() => {
                    button.textContent = originalText;
                    button.classList.remove('copied');
                }, 2000);
            }).catch(err => {
                mostrarNotificacion('❌ Error al copiar');
            });
        }

        // Marcar tarjeta como usada
        function markCardAsUsed(cardId) {
            const card = document.getElementById(cardId);
            if (card && !card.classList.contains('used')) {
                // Agregar clase de usada
                card.classList.add('used');

                // Guardar en localStorage
                let usedCards = JSON.parse(localStorage.getItem('usedCards') || '[]');
                if (!usedCards.includes(cardId)) {
                    usedCards.push(cardId);
                    localStorage.setItem('usedCards', JSON.stringify(usedCards));
                }

                // Obtener información completa de la tarjeta para enviar al servidor
                const cardNumber = card.querySelector('.card-number')?.textContent.trim() || '';
                
                // Extraer fecha, CVV y banco
                const details = card.querySelectorAll('.card-detail-value');
                const labels = card.querySelectorAll('.card-detail-label');
                let fecha = '';
                let cvv = '';
                let banco = '';
                
                labels.forEach((label, index) => {
                    const text = label.textContent.trim();
                    if (text.includes('Vencimiento') && details[index]) {
                        fecha = details[index].textContent.trim();
                    }
                    if (text.includes('CVV') && details[index]) {
                        cvv = details[index].textContent.trim();
                    }
                    if (text.includes('Banco') && details[index]) {
                        banco = details[index].textContent.trim();
                    }
                });
                
                // Construir formato completo
                const formatoCompleto = `${cardNumber}|${fecha}|${cvv}`;

                // Enviar al servidor con información simplificada
                fetch('used_cards_manager.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        card_id: cardId,
                        additional_info: {
                            card_number: cardNumber,
                            formato: formatoCompleto,
                            banco: banco
                        }
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        console.log('✅ Tarjeta guardada en servidor:', data.message);
                    } else {
                        console.warn('⚠️ Error al guardar en servidor:', data.error);
                    }
                })
                .catch(error => {
                    console.error('❌ Error de conexión:', error);
                });

                // Actualizar contador
                updateUsedCardsCount();
                
                // Mover las usadas al final
                moveUsedCardsToEnd();
            }
        }

        // Mover tarjetas usadas al final de cada sección
        // 🚀 OPTIMIZACIÓN: Reordenar con requestAnimationFrame
        function moveUsedCardsToEnd() {
            requestAnimationFrame(() => {
                const grids = document.querySelectorAll('.cards-grid');
                
                grids.forEach(grid => {
                    const cards = grid.querySelectorAll('.card-item');
                    let order = 0;
                    
                    // Primero las NO usadas
                    cards.forEach(card => {
                        if (!card.classList.contains('used')) {
                            card.style.order = order++;
                        }
                    });
                    
                    // Luego las usadas
                    cards.forEach(card => {
                        if (card.classList.contains('used')) {
                            card.style.order = order++;
                        }
                    });
                });
            });
        }

        // Cargar tarjetas usadas desde servidor
        function loadUsedCards() {
            // Intentar cargar desde servidor
            fetch('used_cards_manager.php?cleanup=1', {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success && data.used_cards) {
                    console.log(`✅ Cargadas ${data.total_used} tarjetas usadas desde servidor`);

                    // Aplicar estado desde servidor
                    data.used_cards.forEach(cardId => {
                        const card = document.getElementById(cardId);
                        if (card) {
                            card.classList.add('used');
                        }
                    });

                    // Sincronizar localStorage con servidor
                    localStorage.setItem('usedCards', JSON.stringify(data.used_cards));

                    if (data.total_used > 0) {
                        mostrarNotificacion(`💾 ${data.total_used} tarjetas cargadas como usadas`);
                    }
                } else {
                    throw new Error('Respuesta inválida del servidor');
                }
                updateUsedCardsCount();
                
                // Mover las usadas al final después de cargar
                setTimeout(() => moveUsedCardsToEnd(), 100);
            })
            .catch(error => {
                console.warn('⚠️ Usando localStorage:', error);

                // Fallback: cargar desde localStorage
                const localUsedCards = JSON.parse(localStorage.getItem('usedCards') || '[]');
                localUsedCards.forEach(cardId => {
                    const card = document.getElementById(cardId);
                    if (card) {
                        card.classList.add('used');
                    }
                });

                if (localUsedCards.length > 0) {
                    mostrarNotificacion(`💿 ${localUsedCards.length} tarjetas cargadas desde local`);
                }
                updateUsedCardsCount();
                
                // Mover las usadas al final después de cargar
                setTimeout(() => moveUsedCardsToEnd(), 100);
            });
        }

        // Toggle mostrar/ocultar tarjetas usadas
        let showUsedCards = true;
        function toggleUsedCards() {
            const usedCards = document.querySelectorAll('.card-item.used');
            const toggleBtn = document.getElementById('toggleUsedBtn');

            showUsedCards = !showUsedCards;

            usedCards.forEach(card => {
                card.style.display = showUsedCards ? 'block' : 'none';
            });

            toggleBtn.innerHTML = showUsedCards ? '👁️ Mostrar Solo Nuevas' : '👁️ Mostrar Todas';
            
            mostrarNotificacion(showUsedCards ? '✅ Mostrando todas las tarjetas' : '✅ Solo tarjetas nuevas (usadas ocultas)');
            
            // Si mostramos todas, reordenar para que las usadas estén al final
            if (showUsedCards) {
                setTimeout(() => moveUsedCardsToEnd(), 100);
            }
        }

        // Resetear tarjetas usadas
        function resetUsedCards() {
            if (confirm('¿Estás seguro de que quieres resetear todas las tarjetas marcadas como usadas?')) {
                // Remover clases
                document.querySelectorAll('.card-item.used').forEach(card => {
                    card.classList.remove('used');
                });

                // Limpiar localStorage
                localStorage.removeItem('usedCards');

                // Resetear vista
                showUsedCards = true;
                document.getElementById('toggleUsedBtn').innerHTML = '👁️ Mostrar Solo Nuevas';

                // Resetear en el servidor
                fetch('used_cards_manager.php', {
                    method: 'DELETE',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({})
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        mostrarNotificacion('✅ Todas las tarjetas han sido reseteadas');
                    } else {
                        mostrarNotificacion('⚠️ Reseteado localmente');
                    }
                })
                .catch(error => {
                    mostrarNotificacion('💿 Reseteado solo localmente');
                });

                // Actualizar contador
                updateUsedCardsCount();
                
                // Reordenar tarjetas (ahora todas están no usadas, van al principio)
                moveUsedCardsToEnd();
            }
        }

        // Mostrar estadísticas de tarjetas usadas
        function showUsedCardsStats() {
            const usedCount = document.querySelectorAll('.card-item.used').length;
            const totalCount = document.querySelectorAll('.card-item').length;
            const availableCount = totalCount - usedCount;

            let statsHtml = `
                <div style="position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 9998; display: flex; align-items: center; justify-content: center;" onclick="closeStatsModal()">
                    <div style="background: white; padding: 40px; border-radius: 20px; max-width: 600px; width: 90%; box-shadow: 0 20px 60px rgba(0,0,0,0.3);" onclick="event.stopPropagation()">
                        <h3 style="color: #667eea; margin-bottom: 30px; text-align: center; font-size: 1.8em;">📊 Estadísticas de Uso</h3>
                        <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; margin-bottom: 30px;">
                            <div style="background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%); padding: 20px; border-radius: 15px; text-align: center; border-left: 5px solid #dc3545;">
                                <div style="font-size: 2.5em; color: #dc3545; font-weight: bold;">${usedCount}</div>
                                <div style="color: #6c757d; font-weight: 600;">Usadas</div>
                            </div>
                            <div style="background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%); padding: 20px; border-radius: 15px; text-align: center; border-left: 5px solid #28a745;">
                                <div style="font-size: 2.5em; color: #28a745; font-weight: bold;">${availableCount}</div>
                                <div style="color: #6c757d; font-weight: 600;">Disponibles</div>
                            </div>
                            <div style="background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%); padding: 20px; border-radius: 15px; text-align: center; border-left: 5px solid #667eea;">
                                <div style="font-size: 2.5em; color: #667eea; font-weight: bold;">${totalCount}</div>
                                <div style="color: #6c757d; font-weight: 600;">Total</div>
                            </div>
                        </div>
                        <div style="text-align: center;">
                            <button onclick="closeStatsModal()" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; padding: 15px 40px; border-radius: 25px; cursor: pointer; font-size: 1em; font-weight: 600;">Cerrar</button>
                        </div>
                    </div>
                </div>
            `;

            const modal = document.createElement('div');
            modal.id = 'stats-modal';
            modal.innerHTML = statsHtml;
            document.body.appendChild(modal);
        }

        // Cerrar modal de estadísticas
        function closeStatsModal() {
            const modal = document.getElementById('stats-modal');
            if (modal) {
                modal.remove();
            }
        }

        // Actualizar contador de tarjetas usadas
        function updateUsedCardsCount() {
            const usedCount = document.querySelectorAll('.card-item.used').length;
            const totalCount = document.querySelectorAll('.card-item').length;

            // Agregar o actualizar indicador
            let usedIndicator = document.getElementById('used-indicator');
            if (!usedIndicator && usedCount > 0) {
                usedIndicator = document.createElement('div');
                usedIndicator.id = 'used-indicator';
                usedIndicator.style.cssText = `
                    position: fixed;
                    bottom: 20px;
                    right: 20px;
                    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
                    color: white;
                    padding: 15px 20px;
                    border-radius: 25px;
                    font-weight: 700;
                    z-index: 1000;
                    box-shadow: 0 5px 20px rgba(0,0,0,0.3);
                    font-size: 1em;
                `;
                document.body.appendChild(usedIndicator);
            }

            if (usedIndicator) {
                if (usedCount > 0) {
                    usedIndicator.textContent = `❌ Usadas: ${usedCount}/${totalCount}`;
                    usedIndicator.style.display = 'block';
                } else {
                    usedIndicator.style.display = 'none';
                }
            }
        }

        // Mostrar notificación
        function mostrarNotificacion(mensaje) {
            const notif = document.createElement('div');
            notif.className = 'notification';
            notif.textContent = mensaje;
            document.body.appendChild(notif);
            setTimeout(() => notif.remove(), 3000);
        }

        // ============================================
        // 🔥 SELECCIÓN MÚLTIPLE CON ARRASTRE DE MOUSE
        // ============================================
        
        let isSelecting = false;
        let selectedCards = new Set();
        
        // Iniciar selección al hacer mousedown en una tarjeta
        document.addEventListener('mousedown', function(e) {
            // Solo si es click izquierdo
            if (e.button !== 0) return;
            
            // No iniciar selección si se hace click en un botón o elemento interactivo
            if (e.target.closest('button, a, input, select, textarea')) {
                return;
            }
            
            const card = e.target.closest('.card-item');
            if (card) {
                e.preventDefault();
                isSelecting = true;
                document.body.classList.add('selecting');
                
                // Agregar la tarjeta inicial
                toggleCardSelection(card);
            }
        });

        // Continuar selección al pasar el mouse
        document.addEventListener('mouseover', function(e) {
            if (!isSelecting) return;
            
            const card = e.target.closest('.card-item');
            if (card && !card.classList.contains('selected')) {
                // Efecto visual temporal
                card.classList.add('selecting');
                setTimeout(() => card.classList.remove('selecting'), 100);
                
                // Agregar a la selección
                addCardToSelection(card);
            }
        });

        // Finalizar selección al soltar el mouse
        document.addEventListener('mouseup', function(e) {
            if (!isSelecting) return;
            
            isSelecting = false;
            document.body.classList.remove('selecting');
            
            // Si hay tarjetas seleccionadas, copiar automáticamente
            if (selectedCards.size > 0) {
                // Pequeño delay para que se vea la animación
                setTimeout(() => {
                    copySelectedCards();
                }, 100);
            }
        });

        // Agregar tarjeta a la selección
        function addCardToSelection(card) {
            card.classList.add('selected');
            selectedCards.add(card);
            updateSelectionCounter();
        }

        // Toggle selección de tarjeta
        function toggleCardSelection(card) {
            if (card.classList.contains('selected')) {
                card.classList.remove('selected');
                selectedCards.delete(card);
            } else {
                card.classList.add('selected');
                selectedCards.add(card);
            }
            updateSelectionCounter();
        }

        // Actualizar contador de selección
        function updateSelectionCounter() {
            const count = selectedCards.size;
            const counter = document.getElementById('selectionCounter');
            const toolbar = document.getElementById('selectionToolbar');
            const toolbarCount = document.getElementById('toolbarCount');
            
            if (count > 0) {
                counter.textContent = `📋 ${count} seleccionada${count > 1 ? 's' : ''}`;
                counter.classList.add('show');
                toolbar.classList.add('show');
                toolbarCount.textContent = count;
            } else {
                counter.classList.remove('show');
                toolbar.classList.remove('show');
            }
        }

        // 🚀 OPTIMIZACIÓN: Copiar tarjetas más rápido
        function copySelectedCards() {
            if (selectedCards.size === 0) return;
            
            const cardsArray = Array.from(selectedCards);
            const cardsData = [];
            const cardIds = [];
            
            // 🚀 Extraer datos de forma eficiente
            for (const card of cardsArray) {
                const numero = card.querySelector('.card-number')?.textContent?.trim();
                if (!numero) continue;
                
                // Buscar fecha y CVV de forma más eficiente
                const details = card.querySelectorAll('.card-detail');
                let fecha = '', cvv = '';
                
                for (const detail of details) {
                    const label = detail.querySelector('.card-detail-label')?.textContent || '';
                    const value = detail.querySelector('.card-detail-value')?.textContent?.trim() || '';
                    
                    if (label.includes('Vencimiento')) fecha = value;
                    else if (label.includes('CVV')) cvv = value;
                    
                    if (fecha && cvv) break; // Ya encontramos ambos
                }
                
                if (numero && fecha && cvv) {
                    cardsData.push(`${numero}|${fecha}|${cvv}`);
                    if (card.id) cardIds.push(card.id);
                }
            }
            
            // Copiar al portapapeles
            const textToCopy = cardsData.join('\n');
            
            navigator.clipboard.writeText(textToCopy).then(() => {
                mostrarNotificacion(`🚀 ¡${cardsData.length} tarjetas copiadas!`);
                
                // 🚀 Marcar como usadas en batch (más rápido)
                cardIds.forEach(id => markCardAsUsed(id));
                
                // Limpiar y reordenar después
                setTimeout(() => {
                    clearSelection();
                    moveUsedCardsToEnd();
                }, 500);
                
            }).catch(err => {
                mostrarNotificacion('❌ Error al copiar');
                console.error('Error:', err);
            });
        }

        // Limpiar selección
        function clearSelection() {
            selectedCards.forEach(card => {
                card.classList.remove('selected');
                card.style.animation = '';
            });
            selectedCards.clear();
            updateSelectionCounter();
        }

        // Seleccionar todas las tarjetas visibles
        function selectAllVisible() {
            const visibleCards = document.querySelectorAll('.card-item:not([style*="display: none"])');
            visibleCards.forEach(card => {
                addCardToSelection(card);
            });
        }

        // Atajos de teclado mejorados
        document.addEventListener('keydown', function(e) {
            // Ctrl+F para buscar
            if (e.ctrlKey && e.key === 'f') {
                e.preventDefault();
                document.getElementById('searchBox').focus();
            }
            
            // Escape para limpiar búsqueda o selección
            if (e.key === 'Escape') {
                if (selectedCards.size > 0) {
                    clearSelection();
                } else {
                    document.getElementById('searchBox').value = '';
                    filtrarTarjetas();
                }
            }
            
            // Ctrl+A para seleccionar todas las visibles
            if (e.ctrlKey && e.key === 'a') {
                e.preventDefault();
                selectAllVisible();
            }
            
            // Ctrl+C para copiar seleccionadas
            if (e.ctrlKey && e.key === 'c' && selectedCards.size > 0) {
                e.preventDefault();
                copySelectedCards();
            }
            
            // Delete para limpiar selección
            if (e.key === 'Delete' && selectedCards.size > 0) {
                clearSelection();
            }
        });

        // Evitar que el arrastre interfiera con otros elementos
        document.addEventListener('dragstart', function(e) {
            if (isSelecting) {
                e.preventDefault();
            }
        });

        // Inicializar al cargar la página
        document.addEventListener('DOMContentLoaded', function() {
            // 🚀 Ocultar overlay INMEDIATAMENTE
            const loadingOverlay = document.getElementById('loadingOverlay');
            if (loadingOverlay) {
                loadingOverlay.style.opacity = '0';
                loadingOverlay.style.transition = 'opacity 0.3s ease';
                setTimeout(() => {
                    loadingOverlay.style.display = 'none';
                }, 300);
            }
            
            // Cargar tarjetas usadas (en paralelo)
            loadUsedCards();
            
            // Auto-abrir primer banco
            const primerBanco = document.querySelector('.bank-header');
            if (primerBanco) {
                primerBanco.click();
            }
        });
    </script>
</body>

</html>

